<?php

namespace App\Http\Controllers;

use App\Models\LeaveRequests;
use App\Models\LeaveType;
use App\Models\Notification;
use App\Models\Team;
use App\Models\TeamToMember;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;

class LeaveRequestController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(){
        $leave_requests = '';

        //Update notification read status
        if(isset($_REQUEST['notification_id'])){
            $notification = Notification::find($_REQUEST['notification_id']);

            if (!$notification->is_read){
                $notification->update([
                    'is_read' => 1,
                ]);
            }
        }

        if(auth()->user()->is_admin || auth()->user()->findRole->name === 'HR'){
            $leave_requests = LeaveRequests::orderBy('id','desc')->get();
        }elseif(!auth()->user()->isTeamLead->isEmpty()){
            $team_ids  = auth()->user()->isTeamLead->pluck('id');
            $teamToMember = new TeamToMember();
            $team_users = $teamToMember->getMembersByTeamId($team_ids)->pluck('user_id');

            $leave_requests = LeaveRequests::whereIn('user_id',$team_users)->get();

        }else{
            $leave_requests = LeaveRequests::where('user_id',auth()->id())->get();
        }

        return view('leave_requests.index',compact('leave_requests'));
    }

    public function create(){
        $leave_types = LeaveType::where('status',1)->get();
        return view('leave_requests.create',compact('leave_types'));
    }

    public function store(Request $request){
        $request->validate([
            'leave_type_id'     => 'required',
            'leave_from_date'   => 'required',
            'leave_end_date'    => 'required',
            'reason'            => 'required',
        ]);
        $leave_start = Carbon::create($request->leave_from_date);
        $leave_end = Carbon::create($request->leave_end_date);
        $number_of_days = $leave_start->diffInDays($leave_end) + 1;

        $request->merge([
            'number_of_days' => $number_of_days,
            'leave_from_date' => $leave_start,
            'leave_end_date' => $leave_end,
            'user_id' => auth()->id(),
        ]);

        LeaveRequests::create($request->all());

        //Generate notification for TL and HR
        $to_user_id = [];
        $hr_ids = User::select('users.id')
            ->join('roles', 'users.role_id', '=', 'roles.id')
            ->where('roles.name', "HR")
            ->get();

        $user_id = auth()->id();
        $team_id = TeamToMember::where('user_id',$user_id)->first();
        $th_id = Team::where('id',$team_id->team_id)->get('team_head_user_id');

        foreach ($hr_ids as $id){
            array_push($to_user_id,$id->id);
        }

        foreach ($th_id as $id){
            array_push($to_user_id,$id->team_head_user_id);
        }

        $notificationModel = new Notification();
        $notification_details = [
            'to_user_id' => $to_user_id,
            'from_user_id' => auth()->id(),
            'subject' => 'has requested for leave',
            'is_read' => 0,
            'route' => 'notification',
        ];
        $notificationModel->createNotification($notification_details);
        //Notification generated

        return redirect()->route('leave-management')
            ->with('mesg','Leave request submitted successfully.');
    }

    public function edit($id){
        $id = base64_decode($id);
        $leave_request = LeaveRequests::find($id);
        $leave_types = LeaveType::where('status',1)->get();
        if ($leave_request){
            return view('leave_requests.edit',compact('leave_types','leave_request'));
        }else{
            return redirect()->back()->with('error','Leave request not found.');
        }
    }

    public function update(Request $request){
        $request->validate([
            'leave_type_id'     => 'required',
            'leave_from_date'   => 'required',
            'leave_end_date'    => 'required',
            'reason'            => 'required',
        ]);

        $leave_start = Carbon::create($request->leave_from_date);
        $leave_end = Carbon::create($request->leave_end_date);
        $number_of_days = $leave_start->diffInDays($leave_end) + 1;

        $request->merge([
            'id' => base64_decode($request->id),
            'number_of_days' => $number_of_days,
            'leave_from_date' => $leave_start,
            'leave_end_date' => $leave_end,
            'updated_at' => Carbon::now(),
        ]);

        $leave_request = LeaveRequests::find($request->id);
        if ($leave_request){
            $leave_request->update($request->all());
            return redirect()->route('leave-management')->with('mesg','Leave request updated successfully.');
        }else{
            return redirect()->back()->with('error','Leave request not found.');
        }
    }

    public function delete($id){
        $id = base64_decode($id);
        $leave_request = LeaveRequests::find($id);
        if ($leave_request){
            $leave_request->delete();
            return response('Leave request deleted successfully.');
        }else{
            return response('Leave request not found.');
        }
    }

    public function updateHRStatus(Request $request){
        $request->validate([
            'id' => 'required',
            'action' => 'required',
        ]);

        $status = '';
        $column = '';
        $value  = '';
        $res = '';

        if(auth()->user()->is_admin == true || auth()->user()->findRole->name == 'HR'){
            $column = 'approved_by_hr';
        }else{
            $column = 'approved_by_team_lead';
        }

        if($request->action == 'approve'){
            $status = 1;
            $res = 'Approved successfully.';
        }else{
            $status = 0;
            $res = 'Rejected successfully.';
        }

        try {
            $leave_request = LeaveRequests::findOrFail($request->id);

            $leave_request->update([
                $column => $status,
                'updated_at' => Carbon::now(),
            ]);

            if ($leave_request->approved_by_hr && $leave_request->approved_by_team_lead){
                $leave_request->status = 'Approved';
                $leave_request->save();
            }

            //Notify user for leave request update
            $notificationModel = new Notification();
            $notification_details = [
                'to_user_id' => [$leave_request->user_id],
                'from_user_id' => auth()->id(),
                'subject' => 'Leave Request status updated',
                'is_read' => 0,
                'route' => 'leave-management',
            ];

            $notificationModel->createNotification($notification_details);

            $error = false;
            $message = $res;
        } catch (\Exception $er) {
            $message = $er->getMessage();
            $error = true;
        }

        return ['error' => $error, 'action' => $column, 'message' => $message ];
    }
}
