<?php

namespace App\Http\Controllers;

use App\Models\Loan;
use App\Models\Notification;
use App\Models\NotificationSend;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LoanController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(){
        if (auth()->user()->is_admin){
            $loans = Loan::where('status', '<>', 'Pending')->orderBy('id', 'DESC')->get();
            return view('loan.index_admin', compact('loans'));
        }else{
            $loans = Loan::where('status', '<>', 'Pending')->where('user_id', auth()->id())->orderBy('id', 'DESC')->get();
            return view('loan.index', compact('loans'));
        }
    }

    public function indexPending(){
        if (auth()->user()->is_admin){
            $loans = Loan::where('status', 'Pending')->orderBy('id', 'DESC')->get();
            return view('loan.index_admin_pending', compact('loans'));
        }else{
            $loans = Loan::where('status', 'Pending')->where('user_id', auth()->id())->orderBy('id', 'DESC')->get();
            return view('loan.index', compact('loans'));
        }
    }

    public function create(){
        return view('loan.create');
    }

    public function store(Request $request){
        $request->validate([
            'amount_requested' => 'required',
            'description' => 'required',
            'number_of_installments_total' => 'required',
        ]);

        $request->merge([
            'description' => serialize($request->description),
            'user_id' => auth()->id(),
        ]);

        $loan = Loan::create($request->all());

        //Notification
        if ($loan){
            $notification = Notification::create([
                'subject' => ' has requested for loan',
                'description' => auth()->user()->name.' has requested for loan of Rs: '.$loan->amount_requested,
                'from_user_id' => auth()->id(),
                'to_user_id' => 1,
                'notification_type' => 'Loan Request',
                'route' => route('loan.pending'),
            ]);

            $general_notification = [
                'notification_id' => $notification->id,
                'to_send' => 1,
                'is_read' => 0,
            ];
            NotificationSend::create($general_notification);
        }

        return redirect()->route('loan.pending')->with('mesg', 'Loan requested successfully.');
    }

    public function edit($id){
        $loan = Loan::find(base64_decode($id));

        if ($loan){
            $loan_deduction_month = explode('-', $loan->loan_deduction_month);
            if (auth()->user()->is_admin){
                return view('loan.edit_admin', compact('loan', 'loan_deduction_month'));
            }else{
                return view('loan.edit', compact('loan', 'loan_deduction_month'));
            }
        }
        return redirect()->back()->with('error', 'Record not found.');
    }

    public function update(Request $request){
        $loan = Loan::find($request->id);
        if ($loan){
            $request->validate([
                'amount_requested' => 'required',
                'description' => 'required',
                'number_of_installments_total' => 'required',
            ]);

            $request->merge([
                'description' => serialize($request->description),
                'user_id' => auth()->id(),
            ]);
            $loan->update($request->all());

            return redirect()->route('loan.pending')->with('mesg', 'Loan requested updated successfully.');
        }
        return redirect()->back()->with('error', 'Record not found.');
    }

    public function updateByAdmin(Request $request){

        $rules = [
            'id' => 'required',
            'status' => 'required',
        ];

        if ($request->status == 'Accepted'){
            $rules['disbursement_date'] = 'required';
            $rules['amount_accepted'] = 'required';
            $rules['number_of_installments_total'] = 'required|min:1';
            $rules['deduction_month'] = 'required';
            $rules['deduction_year'] = 'required';
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails())
            return back()->withInput($request->all())->withErrors($validator->errors());

        if ($request->status == 'Accepted'){

            $request->merge([
                'approved_by' => auth()->id(),
                'approval_date' => Carbon::now(),
                'disbursement_date' => Carbon::make($request->disbursement_date),
                'installment_amount' => $request->amount_accepted / $request->number_of_installments_total,
                'number_of_installments_remaining' => $request->number_of_installments_total,
                'loan_deduction_month' => $request->deduction_month.'-'.$request->deduction_year,
            ]);
        }

        $loan = Loan::find($request->id);
        if ($loan){

            $loan->update($request->all());

            //Notification
            if ($request->status == 'Accepted'){
                $notification = Notification::create([
                    'subject' => $loan->status.' loan request',
                    'description' => 'Your loan request has been accepted',
                    'from_user_id' => auth()->id(),
                    'to_user_id' => $loan->user_id,
                    'notification_type' => 'Loan Request',
                    'route' => route('loan'),
                ]);
            }elseif ($request->status == 'Rejected'){
                $notification = Notification::create([
                    'subject' => $loan->status.' loan request',
                    'description' => 'Your loan request has been rejected',
                    'from_user_id' => auth()->id(),
                    'to_user_id' => $loan->user_id,
                    'notification_type' => 'Loan Request',
                    'route' => route('loan'),
                ]);
            }

            $general_notification = [
                'notification_id' => $notification->id,
                'to_send' => 1,
                'is_read' => 0,
            ];
            NotificationSend::create($general_notification);

            return redirect()->route('loan')->with('mesg', 'Loan request updated successfully.');
        }
        return back()->with('error', 'Record not found.');
    }

    public function loanDetails($id){
        $loan = Loan::find($id);

        if (!$loan){
            $res['status'] = false;
        }

        if($loan->status == 'Accepted'){
           $status = '<span class="badge badge-success">'.$loan->status.'</span>';
        }elseif($loan->status == 'Disbursed'){
            $status = '<span class="badge badge-primary">'.$loan->status.'</span>';
        }elseif($loan->status == 'In-Process'){
            $status = '<span class="badge badge-warning">'.$loan->status.'</span>';
        }elseif($loan->status == 'Returned'){
            $status = '<span class="badge badge-success">'.$loan->status.'</span>';
        }elseif($loan->status == 'Pending'){
            $status = '<span class="badge badge-warning">'.$loan->status.'</span>';
        }else{
            $status = '<span class="badge badge-danger">'.$loan->status.'</span>';
        }

        $html = '<div class="col-md-3 col-sm-3"><b>Accepted Amount</b></div>
                 <div class="col-md-3 col-sm-3"><b>: </b>'.(number_format($loan->amount_accepted??0)).'</div>

                 <div class="col-md-3 col-sm-3"><b>Returned Amount</b></div>
                 <div class="col-md-3 col-sm-3"><b>: </b>'.(number_format($loan->amount_returned??0)).'</div>

                 <div class="col-md-3 col-sm-3"><b>Remaining Amount</b></div>
                 <div class="col-md-3 col-sm-3"><b>: </b>'.(number_format($loan->amount_accepted - $loan->amount_returned)).'</div>

                 <div class="col-md-3 col-sm-3"><b>Installment Amount :</b></div>
                 <div class="col-md-3 col-sm-3"><b>: </b>'.(number_format($loan->installment_amount??0)).'</div>

                 <div class="col-md-3 col-sm-3"><b>Total Installments</b></div>
                 <div class="col-md-3 col-sm-3"><b>: </b>'.$loan->number_of_installments_total.'</div>

                 <div class="col-md-3 col-sm-3"><b>Remaining Installments</b></div>
                 <div class="col-md-3 col-sm-3"><b>: </b>'.($loan->number_of_installments_remaining??0).'</div>

                 <div class="col-md-3 col-sm-3"><b>Disbursement Date</b></div>
                 <div class="col-md-3 col-sm-3"><b>: </b>'.($loan->disbursement_date?Carbon::make($loan->disbursement_date)->format('m-d-Y'):'N/A').'</div>

                 <div class="col-md-3 col-sm-3"><b>Status :</b></div>
                 <div class="col-md-3 col-sm-3"><b>: </b>'.$status.'</div>

                 <div class="col-md-3 col-sm-3"><b>Description :</b></div>
                 <div class="col-md-12 col-sm-12">'.unserialize($loan->description).'</div>';

        $res['status'] = true;
        $res['html']   = $html;
        $res['title']  = ($loan->findUser->name??'').' ( Requested Amount : '.$loan->amount_requested.' )';

        return response($res);
    }

    public function delete($id){
        $id = base64_decode($id);
        $loan = Loan::find($id);
        if ($loan){
            $loan->delete();
            return response('Loan request deleted successfully.');
        }else{
            return response('Record not found.');
        }
    }
}
