<?php

namespace App\Http\Controllers;

use App\Models\Interview;
use App\Models\JobTitle;
use App\Models\UserDesignation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Mail;

class InterviewController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(){
        return view('interviews.index');
    }

    public function create(){
        $job_titles = JobTitle::where('status', 1)->orderBy('title')->get();
        $department = UserDesignation::where('status', 1)->orderBy('title')->get();
        $status = ['Called For Test','Test Cleared','Called For Interview','Interview Cleared','Selected', 'Rejected'];

        return view('interviews.create',compact('status','job_titles','department'));
    }

    public function store(Request $request){

        $request->validate([
            'name'             => 'required|string',
            'fathername'       => 'required|string',
            'email'            => 'required|email',
            'resume_cv'        =>  'required|mimes:pdf,doc,docx',
            'cover_letter'     => 'mimes:pdf,doc,docx',
        ]);

        if ($request->hasFile('resume_cv')){
            $cv_file = $request->resume_cv->getClientOriginalName();
            $request->resume_cv->move(public_path('interview/resume'), $cv_file);
            $request->merge(['resume' => $cv_file]);
        }

        if ($request->hasFile('cover_attachment')){
            $cover_letter = $request->cover_attachment->getClientOriginalName();
            $request->cover_attachment->move(public_path('interview/coverLetter'), $cover_letter);
            $request->merge(['cover_letter' => $cover_letter]);
        }

        $status = json_encode($request->status);
        $referance_data = array();
        $temp = array();
        if (is_array($request->r_name)){
            foreach ($request->r_name as $key=>$value){
                $referance_data['r_name']    = $value;
                $referance_data['r_contact'] = $request->r_contact[$key];
                $referance_data['r_relation'] = $request->r_relation[$key];
                $referance_data['r_company'] = $request->r_company[$key];
                $referance_data['r_position'] = $request->r_position[$key];

                $temp[] = $referance_data;
            }
            $referance = serialize($temp);
            $request->merge(['referance' => $referance]);
        }

        $request->merge(['status' => $status]);

        Interview::create($request->all());

        $status  = json_decode($request->status);

        if ($status[0] == 'Called For Interview'){

            $mesg_html = "<p style='font-size: 12px'>Your Interview $request->interview_schedule</p>";

            Mail::send([],[], function($message) use($mesg_html, $request)
            {
                $message->subject('Called For Interview')->setBody($mesg_html, 'text/html');
                $message->to($request->email, $request->name);
                $message->cc('charlesxavier.mh2022@gmail.com');
            });

        }elseif($status[0] == 'Called For Test'){

            $mesg_html =  "<p style='font-size: 12px'>Your Test $request->interview_schedule</p>";

            Mail::send([],[], function($message) use($mesg_html, $request)
            {
                $message->subject('Called For Test')->setBody($mesg_html, 'text/html');
                $message->to($request->email, $request->name);
                $message->cc('charlesxavier.mh2022@gmail.com');

            });

        }

        if ($request->create_employee){

            $name = explode(' ', $request->name);
            $data['firstname'] = $name[0]??'';
            $data['lastname'] = $name[1]??'';
            $data['name'] = $request->name;
            $data['email'] = $request->email;
            $data['phone'] = $request->phone??'';
            $data['gender'] = $request->gender??'';
            $data['marital_status'] = $request->marital_status??'';
            $data['joining_date'] = $request->joining_date??'';
            $data['designation_id'] = $request->department_id??'';
            $data['job_title_id'] = $request->job_title_id??'';

            return redirect()->route('create.user')->with('mesg','Interview created successfully.')->with('data', $data);
        }else{
            return redirect()->route('interview')->with('mesg','Interview created successfully.');
        }
    }

    public function showAllInterviewList(Request $request){

        $columns_list = array(
            0 =>'id',
            1 =>'name',
            2 =>'fathername',
            3=> 'email',
            6=>'department',
            8=>'applied_for',
            7=> 'phone',
            16=> 'status_id',
            11=>'resume',
            12=>'cover_letter',
        );

        $totalDataRecord = Interview::whereIn('id', function($query) {
            $query->from('interviews')->groupBy('email')->selectRaw('MAX(id)');
        })->count();
        $totalFilteredRecord = $totalDataRecord;

        $limit_val = $request->input('length');
        $start_val = $request->input('start');
        $order = 'interviews.'.$columns_list[$request->input('order.0.column')];
        $dir_val = 'desc';

        if(empty($request->input('search.value')))
        {
            $post_data = Interview::whereIn('id', function($query) {
                $query->from('interviews')->groupBy('email')->selectRaw('MAX(id)');
            })
                ->offset($start_val)
                ->limit($limit_val)
                ->orderBy($order,$dir_val)
                ->get();
        }
        else {
            $search_text = $request->input('search.value');

            $post_data = Interview::where('id','LIKE',"%{$search_text}%")
                ->orWhere('email', 'LIKE',"%{$search_text}%")
                ->orWhere('name', 'LIKE',"%{$search_text}%")
                ->whereIn('id', function($query) {
                    $query->from('interviews')->groupBy('email')->selectRaw('MAX(id)');
                })
                ->offset($start_val)
                ->limit($limit_val)
                ->orderBy($order,$dir_val)
                ->get();

            $totalFilteredRecord = Interview::whereIn('id', function($query) {
                $query->from('interviews')->groupBy('email')->selectRaw('MAX(id)');
            })->where('id','LIKE',"%{$search_text}%")
                ->orWhere('name', 'LIKE',"%{$search_text}%")
                ->orWhere('fathername', 'LIKE',"%{$search_text}%")
                ->orWhere('email', 'LIKE',"%{$search_text}%")
                ->orWhere('phone', 'LIKE',"%{$search_text}%")
                ->count();
        }

        $data_val = array();
        if(!empty($post_data))
        {
            $i=0;
            $file_type_cv = '';
            $file_type_letter ='';
            $interview_model = new Interview();

            foreach ($post_data as $post_val)
            {
                $post_status = json_decode($post_val->status);
                $status_html = $interview_model->statusValueColor($post_status);

                if (!empty($post_val->resume)){
                    $file_type_cv = explode(".",$post_val->resume)[1];
                }

                if (!empty($post_val->cover_letter)){
                    $file_type_letter = explode(".",$post_val->cover_letter)[1];
                }

                $i++;
                $deletedata = $post_val->id;
                $id     =  base64_encode($deletedata);
                $email  =  base64_encode($post_val->email);
                $dataedit =  route('edit.interview',['id'=>$id]);
                $showdata = route('show.interview',['id'=>$id,'email'=>$email]);

                if ($file_type_cv == 'docx' || $file_type_cv == 'DOCX' || $file_type_cv == 'doc') {
                    $urlPathCv = "https://view.officeapps.live.com/op/embed.aspx?src=".urlencode(\URL::to('/') . '/public/interview/resume/'.$post_val->resume);
                }
                else{
                    $urlPathCv = asset('/interview/resume/'.$post_val->resume);
                }

                if ($file_type_letter == 'docx' || $file_type_letter == 'DOCX' || $file_type_letter == 'doc') {
                    $urlPathLetter = "https://view.officeapps.live.com/op/embed.aspx?src=".urlencode(\URL::to('/') . '/public/interview/coverLetter/'.$post_val->cover_letter);
                }
                else{
                    $urlPathLetter = asset('/interview/coverLetter/'.$post_val->resume);
                }

                $cv = (!empty($post_val->resume))?"<a class='btn btn-success text-white' href='".$urlPathCv."' class='fancybox' data-fancybox data-type='iframe' data-preload='false' >View Cv</a>":"N/A";
                $cover_letter = (!empty($post_val->cover_letter))?"<a class='btn btn-success text-white' href='".$urlPathLetter."' class='fancybox' data-fancybox data-type='iframe' data-preload='false' >View Letter</a>":"N/A";

//                $delete_interview = Gate::allows("delete.interview")?"<button class='btn btn-danger delete-btn' data-id='{$id}' title='Delete'><i class='fa fa-trash'></i> </button>":"";
//                $edit_interview = Gate::allows("edit.interview")?"<a class='btn btn-primary btn-style' href='{$dataedit}' title='Edit'><i class='fa fa-edit'></i> </a>":"";

                $show_interview = Gate::allows("show.user")?"<a class='btn btn-success btn-style' href='{$showdata}' title='View'><i class='fa fa-eye'></i> </a>":"";

                $postnestedData['id'] = $i;
                $postnestedData['name'] = $post_val->name;
                $postnestedData['fathername'] = $post_val->fathername??'N/A';
                $postnestedData['email'] = $post_val->email;
                $postnestedData['phone'] = $post_val->phone??'N/A';
                $postnestedData['department'] = $post_val->findDepartment->title??'N/A';
                $postnestedData['applied_for'] = $post_val->findJobTitle->title??'N/A';
                $postnestedData['resume'] =$cv??'N/A';
                $postnestedData['cover_letter'] = $cover_letter??'N/A';
                $postnestedData['status_id'] =$status_html??'N/A';
                $postnestedData['options'] = "<div class='btn-group'>{$show_interview}
                                                <a class='btn btn-primary btn-style' href='{$dataedit}' title='Edit'><i class='fa fa-edit'></i> </a>
                                              </div>";

                $data_val[] = $postnestedData;
            }
        }
        $draw_val = $request->input('draw');
        $get_json_data = array(
            "draw"            => intval($draw_val),
            "recordsTotal"    => intval($totalDataRecord),
            "recordsFiltered" => intval($totalFilteredRecord),
            "data"            => $data_val
        );

        echo json_encode($get_json_data);
    }

    public function edit($id){
        $id =    base64_decode($id);
        $interview = Interview::find($id);
        $job_titles = JobTitle::where('status', 1)->orderBy('title')->get();
        $department = UserDesignation::where('status', 1)->orderBy('title')->get();
        $status = ['Called For Test','Test Cleared','Called For Interview','Interview Cleared','Selected', 'Rejected'];

        $referance = unserialize($interview->referance);

        return view('interviews.edit',compact('interview','status','referance','job_titles','department'));

    }

    public function update(Request $request){

        $email  =  base64_encode($request->email);

        $request->validate([
            'name'             => 'required|string',
            'fathername'       => 'required|string',
            'email'            => 'required|email',
            'resume_cv'        =>  'mimes:pdf,doc,docx',
            'cover_letter'     => 'mimes:pdf,doc,docx',
        ]);

        $request->merge(['id' => base64_decode($request->id)]);
        $interview = Interview::find($request->id);

        if($interview){

            if ($request->hasFile('resume_cv')){
                $cv_file = $request->resume_cv->getClientOriginalName();
                $request->resume_cv->move(public_path('interview/resume'), $cv_file);
                $request->merge(['resume' => $cv_file]);
            }

            if ($request->hasFile('cover_attachment')){
                $cover_letter = $request->cover_attachment->getClientOriginalName();
                $request->cover_attachment->move(public_path('interview/coverLetter'), $cover_letter);
                $request->merge(['cover_letter' => $cover_letter]);
            }

            $status = json_encode($request->status);
            $referance_data = array();
            $temp = array();
            if (is_array($request->r_name)){
                foreach ($request->r_name as $key=>$value){
                    $referance_data['r_name']    = $value;
                    $referance_data['r_contact'] = $request->r_contact[$key];
                    $referance_data['r_relation'] = $request->r_relation[$key];
                    $referance_data['r_company'] = $request->r_company[$key];
                    $referance_data['r_position'] = $request->r_position[$key];

                    $temp[] = $referance_data;
                }
                $referance = serialize($temp);
                $request->merge(['referance' => $referance]);
            }
            $request->merge(['status' => $status]);

            $offer_accept = intval($request->offer_accept_reject);

            $request->merge(['offer_accept_reject' => $offer_accept]);

            $res = $interview->update($request->all());

            $status  = json_decode($interview->status);

            if ($status[0] == 'Selected'){

                $mesg_html =  "<p style='font-size: 12px'>Your Offer Letter $request->name</p>";

                Mail::send([],[], function($message) use($mesg_html, $request)
                {
                    $message->subject('Offer Letter')->setBody($mesg_html, 'text/html');
                    $message->to($request->email, $request->name);
                    $message->cc('charlesxavier.mh2022@gmail.com');
                });
            }

            if ($request->offer_accept_reject == 1){
                $name = explode(' ', $request->name);
                $data['firstname'] = $name[0]??'';
                $data['lastname'] = $name[1]??'';
                $data['name'] = $request->name;
                $data['email'] = $request->email;
                $data['phone'] = $request->phone??'';
                $data['gender'] = $request->gender??'';
                $data['marital_status'] = $request->marital_status??'';
                $data['joining_date'] = $request->joining_date??'';
                $data['designation_id'] = $request->department_id??'';
                $data['job_title_id'] = $request->job_title_id??'';

                return redirect()->route('create.user')->with('mesg','Offer Accepted Successfully.')->with('data', $data);

            }elseif ($res){
                return redirect()->route('show.interview',['id'=>base64_encode($request->id),'email'=>$email])->with('mesg','Interview Record updated successfully.');
                //return redirect()->back();
            }
        }
    }

    public function delete($id){
        $id = base64_decode($id);
        $interview = Interview::find($id);
        if($interview){
            $interview->delete();
            return response('Interview Record deleted successfully.');
        }else{
            return response('Interview Record not found.');
        }
    }

    public function show(Request $request){
        $id    = base64_decode($request['id']);
        $email = base64_decode($request['email']);

        $particular_record  = Interview::where('id',$id)->first();
        $interview  = Interview::where('email',$email)->orderby('id','desc')->get();

        $status_html = 'N/A';
        if (!empty($particular_record)){
            $interview_model = new Interview();
            $post_status =json_decode($particular_record->status);
            $status_html = $interview_model->statusValueColor($post_status);
        }

        $file_type_cv = (!empty($particular_record->resume))?explode(".",$particular_record->resume)[1]:'';
        $file_type_cover =  (!empty($particular_record->cover_letter))?explode(".",$particular_record->cover_letter)[1]:'';

        return view('interviews.show',compact('interview','particular_record','file_type_cv','file_type_cover','status_html'));
    }

    public function  interviewDetail(Request $request){

        $id = $request->id;
        $interview_data = Interview::find($id);
        $post_status = json_decode($interview_data->status);

        $interview_model = new Interview();
        $status_html = $interview_model->statusValueColor($post_status);

        return response([
            'item_data' => $interview_data,
            'status_data' => $status_html,
            'department'  => $interview_data->findDepartment->title,
            'job_title'  => $interview_data->findJobTitle->title,
        ]);
    }
}
